<?php

namespace App\Http\Controllers\API\V4;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Categories;
use App\Restorant;

class CategoryController extends Controller
{
   public function fetchCategories(Request $request)
    {
        $request->validate([
            'UserKey' => 'required|string',
            'LastSyncTime' => 'nullable|integer',
        ]);

        $userKey = $request->input('UserKey');
        $lastSyncTime = $request->input('LastSyncTime');

        // Find user by id_session
        $user = User::where('id_session', $userKey)->first();

        if (!$user) {
            return response()->json([
                'status' => 'true',
                'errCode' => '00',
                'msg' => 'User Not Found',
                'data' => []
            ], 404);
        }

        // Get the master user by phone_number (assuming 'master' stores phone_number)
        $masterUser = User::where('phone_number', $user->master)->first();

        if (!$masterUser) {
            // Master user not found, return empty data or error
            return response()->json([
                'status' => 'true',
                'errCode' => '00',
                'msg' => 'Master User Not Found',
                'data' => []
            ], 404);
        }

        // Base query for categories with status '0' and user = master phone_number
        $query = Categories::select([
                'local_id',
                'id as cloud_id',
                'name',
                'restorant_id as onlinestore_id',
                'user',
                'active',
                'updated_at_long as updated_at'
            ])
            ->where('user', $masterUser->phone_number)
            ->where('status', '0');

        // Filter by lastSyncTime if provided
        if ($lastSyncTime && $lastSyncTime > 1) {
            $query->where('updated_at_long', '>', $lastSyncTime);
        }

        $categories = $query->orderByDesc('id')->get();

        if ($categories->isEmpty()) {
            return response()->json([
                'status' => 'true',
                'errCode' => '02',
                'msg' => 'No Categories Found',
                'data' => []
            ], 200);
        }

        // Return categories with user as the original user's master value
        $data = $categories->map(function ($category) use ($user) {
            return [
                'local_id' => $category->local_id,
                'cloud_id' => $category->cloud_id,
                'name' => $category->name,
                'onlinestore_id' => $category->onlinestore_id,
                'user' => $user->master,
                'active' => $category->active,
                'updated_at' => $category->updated_at_long,
            ];
        });

        return response()->json($data, 200);
    }



    public function syncCategory(Request $request)
    {
        $userKey = $request->query('UserKey');

        if (!$userKey) {
            return response()->json([
                'status' => false,
                'msg' => 'UserKey is required',
            ], 400);
        }

        $user = User::where('id_session', $userKey)->first();

        if (!$user) {
            return response()->json([
                'status' => false,
                'msg' => 'User not found',
            ], 404);
        }

        $data = $request->json()->all();

        $validator = Validator::make($data, [
            'local_id' => 'required|string',
            'cloud_id' => 'nullable|integer',
            'name' => 'required|string',
            'onlinestore_id' => 'nullable|integer',
            'active' => 'required|boolean',
            'user' => 'required|string',
            'updated_at' => 'required|numeric'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'msg' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $company = Restorant::where('userz', $user->master)->first();

        if (!$company) {
            return response()->json([
                'status' => false,
                'msg' => 'Company not found for master user',
            ], 404);
        }

        $cloudId = $data['cloud_id'] ?? 0;

        $existingCategory = Categories::where('local_id', $data['local_id'])
            ->orWhere('id', $cloudId)
            ->first();

        if ($existingCategory) {
            if ($existingCategory->updated_at_long >= $data['updated_at']) {
                return response()->json([
                    [["status" => "success"]]
                ]);
            }

            // Update the existing category
            $existingCategory->update([
                'name' => $data['name'],
                'updated_at_long' => $data['updated_at'],
                'local_id' => $data['local_id'],
            ]);
        } else {
            // Insert new category
            Categories::create([
                'local_id' => $data['local_id'],
                'name' => $data['name'],
                'restorant_id' => $company->id,
                'active' => $data['active'],
                'user' => $data['user'],
                'updated_at_long' => $data['updated_at'],
            ]);
        }

        return response()->json([
            [["status" => "success"]]
        ]);
    }
}
