<?php

namespace App\Http\Controllers\API\V4;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use App\User;
use App\Models\LogsCustomerDebtPayments;

class CustomerController extends Controller
{
     public function fetchCustomers(Request $request)
    {
        $key = $request->query('UserKey');
        $lastSyncTime = $request->query('LastSyncTime');

        $user = DB::table('users')->where('id_session', $key)->first();

        if (!$user) {
            return response()->json([]);
        }

        $masterUser = DB::table('users')->where('phone_number', $user->master)->first();

        if (!$masterUser) {
            return response()->json([]);
        }

        $query = DB::table('customer')->where('user', $masterUser->phone_number);

        if (!empty($lastSyncTime) && $lastSyncTime > 1) {
            $query->where('updated_at', '>', $lastSyncTime);
        }

        $customers = $query->orderBy('name_customer', 'asc')->get();

        return response()->json($customers);
    }

    public function syncCustomer(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'local_id_customer' => 'required|string',
            'id_customer' => 'nullable|string',
            'name_customer' => 'required|string',
            'email' => 'nullable|email',
            'telephone' => 'nullable|string',
            'customercode' => 'nullable|string',
            'address' => 'nullable|string',
            'active' => 'nullable|string',
            'status' => 'nullable|string',
            'debt' => 'nullable|numeric',
            'user' => 'nullable|string',
            'updated_at' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $data = $validator->validated();

        $existingCustomer = DB::table('customer')
            ->where('local_id_customer', $data['local_id_customer'])
            ->orWhere('id_customer', $data['id_customer'] ?? '0')
            ->first();

        if ($existingCustomer) {
            // Skip if existing data is newer
            if (!empty($data['updated_at']) && $existingCustomer->updated_at >= $data['updated_at']) {
                return response()->json([['status' => 'success']]);
            }

            // Update existing record
            DB::table('customer')
                ->where('local_id_customer', $data['local_id_customer'])
                ->orWhere('id_customer', $data['id_customer'] ?? '0')
                ->update([
                    'name_customer' => $data['name_customer'],
                    'local_id_customer' => $data['local_id_customer'],
                    'address' => $data['address'] ?? '',
                    'email' => $data['email'] ?? '',
                    'customercode' => $data['customercode'] ?? '',
                    'debt' => $data['debt'] ?? 0,
                    'status' => $data['status'] ?? '',
                    'telephone' => $data['telephone'] ?? '',
                    'active' => $data['active'] ?? '',
                    'img' => 'avatar.png', 
                    'updated_at' => $data['updated_at'],
                ]);

            return response()->json([['status' => 'success']]);
        } else {
            // Insert new record
            DB::table('customer')->insert([
                'local_id_customer' => $data['local_id_customer'],
                'name_customer' => $data['name_customer'],
                'address' => $data['address'] ?? '',
                'email' => $data['email'] ?? '',
                'telephone' => $data['telephone'] ?? '',
                'customercode' => $data['customercode'] ?? '',
                'img' => 'avatar.png',
                'user' => $data['user'] ?? '',
                'active' => $data['active'] ?? '',
                'status' => $data['status'] ?? '',
                'debt' => $data['debt'] ?? 0,
                'updated_at' => $data['updated_at'],
            ]);

            return response()->json([['status' => 'success']]);
        }
    }


    public function getCustomerDebtHistory(Request $request)
{
    $userKey = $request->query('UserKey');
    $lastSyncTime = $request->query('LastSyncTime');

    $user = DB::table('users')->where('id_session', $userKey)->first();

    if (!$user) {
        return response()->json([
            'status' => 'true',
            'errCode' => '00',
            'msg' => 'User Not Found',
            'data' => []
        ]);
    }

    $masterAccount = DB::table('users')->where('phone_number', $user->master)->first();

    $query = DB::table('customerdebthistory')
        ->where('user', $masterAccount->phone_number);

    if ($lastSyncTime && $lastSyncTime > 1) {
        $query->where('updated_at', '>', $lastSyncTime);
    }

    $debts = $query->orderBy('date', 'asc')->get();

    if ($debts->isEmpty()) {
        return response()->json([
            'status' => 'true',
            'errCode' => '02',
            'msg' => 'No Debt History Found',
            'data' => []
        ]);
    }

    return response()->json($debts);
}


public function syncCustomerDebtHistory(Request $request)
{
    $userKey = $request->query('UserKey');

    $data = $request->all();

    $localId = $data['local_id_customerdebthistory'] ?? null;
    $cloudId = $data['id_customerdebthistory'] ?? null;
    $customerId = $data['id_customer'] ?? null;
    $user = $data['user'] ?? null;
    $invoice = $data['no_invoice'] ?? null;
    $nominal = $data['nominal'] ?? null;
    $status = $data['status'] ?? null;
    $date = $data['date'] ?? null;
    $updatedAt = $data['updated_at'] ?? null;

    $existingUser = DB::table('users')->where('id_session', $userKey)->first();
    if (!$existingUser) {
        return response()->json([['status' => 'error', 'message' => 'Invalid UserKey']]);
    }

    $existingRecord = DB::table('customerdebthistory')
        ->where(function ($query) use ($localId, $cloudId) {
            $query->where('local_id_customerdebthistory', $localId);
            if (!empty($cloudId)) {
                $query->orWhere('id_customerdebthistory', $cloudId);
            }
        })->first();

    if ($existingRecord) {
        if ($updatedAt && $existingRecord->updated_at >= $updatedAt) {
            return response()->json([['status' => 'success']]);
        }

        DB::table('customerdebthistory')
            ->where(function ($query) use ($localId, $cloudId) {
                $query->where('local_id_customerdebthistory', $localId);
                if (!empty($cloudId)) {
                    $query->orWhere('id_customerdebthistory', $cloudId);
                }
            })->update([
                'nominal' => $nominal,
                'status' => $status,
                'updated_at' => $updatedAt,
                'local_id_customerdebthistory' => $localId
            ]);

        return response()->json([['status' => 'success']]);
    } else {
        DB::table('customerdebthistory')->insert([
            'local_id_customerdebthistory' => $localId,
            'id_customer' => $customerId,
            'user' => $user,
            'no_invoice' => $invoice,
            'nominal' => $nominal,
            'date' => $date,
            'status' => $status,
            'updated_at' => $updatedAt
        ]);

        return response()->json([['status' => 'success']]);
    }
}


public function getCustomerDebtPayments(Request $request)
{
    $userKey = $request->query('UserKey');
    $lastSyncTime = $request->query('LastSyncTime');

    $user = User::where('id_session', $userKey)->first();

    if (!$user) {
        return response()->json([
            'status' => 'true',
            'errCode' => '00',
            'msg' => 'User Not Found',
            'data' => []
        ]);
    }

    $master = User::where('phone_number', $user->master)->first();

    if (!$master) {
        return response()->json([
            'status' => 'true',
            'errCode' => '01',
            'msg' => 'Master Account Not Found',
            'data' => []
        ]);
    }

    $query = LogsCustomerDebtPayments::where('user', $master->phone_number);

    if ($lastSyncTime) {
        $query->where('updated_at', '>', $lastSyncTime);
    }

    $logs = $query->get();

    if ($logs->isEmpty()) {
        return response()->json([
            'status' => 'true',
            'errCode' => '02',
            'msg' => 'No Debt Pay History',
            'data' => []
        ]);
    }

    $data = $logs->map(function ($log) {
        return [
            'id_logsCustomerDebtPayments' => $log->id_logscustomerdebtpayments,
            'local_id_logsCusDebtPay' => $log->local_id_logscustomerdebtpayments,
            'id_customer' => $log->id_customer,
            'user' => $log->user,
            'no_invoice' => $log->no_invoice,
            'nominal' => $log->nominal,
            'paymentmethord' => $log->paymentmethord,
            'date' => $log->date,
            'updated_at' => $log->updated_at
        ];
    });

    return response()->json($data);
}

public function syncCustomerDebtPayment(Request $request)
{
    $userKey = $request->query('UserKey');
    $user = User::where('id_session', $userKey)->first();

    if (!$user) {
        return response()->json([
            ['status' => 'User not found']
        ]);
    }

    $data = $request->all();

    $cloudId = $data['id_logsCustomerDebtPayments'] ?? '0';
    $localId = $data['local_id_logsCusDebtPay'] ?? null;
    $updatedAt = $data['updated_at'] ?? null;

    $existing = LogsCustomerDebtPayments::where('local_id_logscustomerdebtpayments', $localId)
        ->orWhere('id_logscustomerdebtpayments', $cloudId)
        ->first();

    if ($existing) {
        if ($updatedAt && $existing->updated_at >= $updatedAt) {
            return response()->json([
                ['status' => 'success']
            ]);
        }

        $existing->update(['updated_at' => $updatedAt]);
        return response()->json([
            ['status' => 'success']
        ]);
    }

    $log = new LogsCustomerDebtPayments();
    $log->local_id_logscustomerdebtpayments = $localId;
    $log->id_customer = $data['id_customer'] ?? null;
    $log->user = $data['user'] ?? null;
    $log->no_invoice = $data['no_invoice'] ?? null;
    $log->nominal = $data['nominal'] ?? null;
    $log->paymentmethord = $data['paymentmethord'] ?? null;
    $log->note = $data['note'] ?? null;
    $log->date = $data['date'] ?? null;
    $log->updated_at = $updatedAt;

    if ($log->save()) {
        return response()->json([
            ['status' => 'success']
        ]);
    }

    return response()->json([
        ['status' => 'error', 'message' => 'Insert failed']
    ]);
}

}
