<?php

namespace App\Http\Controllers\API\V4;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\User;
use App\Models\Expenses;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Carbon\CarbonPeriod;
use Illuminate\Support\Str;

class ExpensesController extends Controller
{
    public function getExpensesHistory(Request $request)
{
    $key = $request->query('key');
    $startDateRaw = $request->query('start_date');
    $finishDateRaw = $request->query('finish_date');
    $status = $request->query('status'); // Optional

    $user = User::where('id_session', $key)->first();

    if (!$user) {
        return response()->json([
            'status' => 'true',
            'errCode' => '00',
            'msg' => 'User Not Found',
        ]);
    }

    // Validate and parse dates
    try {
        $startDate = Carbon::parse($startDateRaw)->startOfDay();
        $finishDate = Carbon::parse($finishDateRaw)->endOfDay();
    } catch (\Exception $e) {
        return response()->json([
            'status' => 'false',
            'errCode' => '02',
            'msg' => 'Invalid date format.',
            'error' => $e->getMessage(),
        ]);
    }

    if ($startDate->greaterThan($finishDate)) {
        return response()->json([
            'status' => 'false',
            'errCode' => '03',
            'msg' => 'Start date must be before or equal to finish date.',
        ]);
    }

    $master = User::where('phone_number', $user->master)->first();

    $response = [
        'status' => 'true',
        'errCode' => '01',
        'msg' => 'Success',
        'data' => []
    ];

    $period = CarbonPeriod::create($startDate, $finishDate);

    foreach ($period as $date) {
        $tanggal = $date->format('Y-m-d');

        if ($user->level === 'master') {
            $nominal = Expenses::where('user', $master->phone_number)
                ->where('date', $tanggal)
                ->where('status', 1)
                ->sum('totalnominal');
        } else {
            $nominal = Expenses::where('operator', $user->phone_number)
                ->where('date', $tanggal)
                ->where('status', 1)
                ->sum('totalnominal');
        }

        $details = [];
        $spendings = Expenses::where('user', $master->phone_number)
            ->where('date', $tanggal)
            ->where('status', 1)
            ->orderBy('id_spending_data', 'asc')
            ->get();

        foreach ($spendings as $spending) {
            $operator = User::where('phone_number', $spending->operator)->first();

            $details[] = [
                'no_invoice' => $spending->no_invoice,
                'date' => $spending->date,
                'nominal' => $spending->totalnominal,
                'oleh' => $operator ? $operator->name : '',
            ];
        }

        $response['data'][] = [
            'totalnominal' => $nominal ?? 0,
            'date' => $tanggal,
            'detail' => $details
        ];
    }

    return response()->json($response);
}

public function storeExpense(Request $request)
    {
        $data = $request->all();

        $tanggal = $data['date'] ?? null;
        $jumlahnya = $data['amount'] ?? null;
        $key = $data['key'] ?? null;

        if (!$tanggal || !$jumlahnya || !$key || !isset($data['spending'])) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'Missing required fields'
            ]);
        }

        // Get user by session key
        $user = DB::table('users')->where('id_session', $key)->first();

        if (!$user) {
            return response()->json([
                'status' => 'true',
                'errCode' => '00',
                'msg' => 'User Not Found'
            ]);
        }

        // Get master account by phone number
        $master = DB::table('users')->where('phone_number', $user->master)->first();

        if (!$master) {
            return response()->json([
                'status' => 'false',
                'errCode' => '02',
                'msg' => 'Master user not found'
            ]);
        }

        // Get last spending_data id
        $lastSpendingData = DB::table('spending_data')
            ->where('user', $master->phone_number)
            ->orderByDesc('id_spending_data')
            ->first();

        $lastId = $lastSpendingData->id_spending_data ?? 1;
        $dateCode = now()->format('dHi');

        // Generate short code from full name
        $initials = collect(explode(' ', $user->name))
            ->map(fn($part) => Str::substr($part, 0, 1))
            ->implode('');

        $kode = strtoupper($initials);
        $no_invoice = "{$kode}-{$lastId}{$dateCode}";

        // Insert into spending table
        foreach ($data['spending'] as $spending) {
            DB::table('spending')->insert([
                'name_spending' => $spending['name_spending'],
                'user' => $master->phone_number,
                'no_invoice' => $no_invoice,
                'nominal' => $spending['nominal'],
                'date' => $tanggal
            ]);
        }

        // Insert into spending_data table
        DB::table('spending_data')->insert([
            'no_invoice' => $no_invoice,
            'date' => $tanggal,
            'operator' => $user->phone_number,
            'totalnominal' => $jumlahnya,
            'id_store' => $user->id_store,
            'user' => $master->phone_number
        ]);

        return response()->json([
            'status' => 'true',
            'errCode' => '01',
            'msg' => 'Success',
            'data' => [
                'invoice' => $no_invoice
            ]
        ]);
    }

    public function getExpenseDetails(Request $request)
{
    $key = $request->query('key');
    $no_invoice = $request->query('no_invoice');

    // Find user by session key
    $user = DB::table('users')->where('id_session', $key)->first();

    if (!$user) {
        return response()->json([
            'status' => 'true',
            'errCode' => '00',
            'msg' => 'User Not Found'
        ]);
    }

    // Get master account
    $master = DB::table('users')->where('phone_number', $user->master)->first();

    if (!$master) {
        return response()->json([
            'status' => 'false',
            'errCode' => '03',
            'msg' => 'Master account not found'
        ]);
    }

    // Fetch spending_data
    $spendingData = DB::table('spending_data')->where('no_invoice', $no_invoice)->first();
    $spendings = DB::table('spending')->where('no_invoice', $no_invoice)->get();

    if ($spendings->isEmpty()) {
        return response()->json([
            'status' => 'true',
            'errCode' => '02',
            'msg' => 'No Expenses'
        ]);
    }

    // Get operator's name
    $operator = DB::table('users')->where('phone_number', $spendingData->operator ?? '')->first();

    $info = [
        'no_invoice'     => $spendingData->no_invoice,
        'oleh'           => $operator->name ?? 'Unknown',
        'date'           => $spendingData->date,
        'totalnominal'   => $spendingData->totalnominal,
    ];

    $pengeluaran = $spendings->map(function ($spending) {
        return [
            'name_spending' => $spending->name_spending,
            'date'          => $spending->date,
            'nominal'       => $spending->nominal,
        ];
    });

    return response()->json([
        'status' => 'true',
        'errCode' => '01',
        'msg' => 'Success',
        'data' => [
            'info' => $info,
            'data' => $pengeluaran,
        ]
    ]);
}


public function markExpenseInactive(Request $request)
{
    $key = $request->query('key');
    $no_invoice = $request->query('no_invoice');

    // Find user by session key
    $user = DB::table('users')->where('id_session', $key)->first();

    if (!$user) {
        return response()->json([
            'status'  => 'true',
            'errCode' => '00',
            'msg'     => 'User Not Found'
        ]);
    }

    // Get master account (if needed for any validation logic)
    $master = DB::table('users')->where('phone_number', $user->master)->first();

    // Update spending_data status
    DB::table('spending_data')
        ->where('no_invoice', $no_invoice)
        ->update(['status' => 0]);

    return response()->json([
        'status'  => 'true',
        'errCode' => '01',
        'msg'     => 'Success' 
    ]);
}
}
