<?php

namespace App\Http\Controllers;

use App\Models\Incident;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;
use Illuminate\View\View;

class HomeController extends Controller
{
    public function index(): View
        {
        // Get date range for current year
        $currentYear = Carbon::now()->year;
        $startOfYear = Carbon::create($currentYear, 1, 1);
        $endOfYear = Carbon::create($currentYear, 12, 31);

        // Basic statistics
        $totalIncidents = Incident::count();
        $thisYearIncidents = Incident::where('year', $currentYear)->count();
        $totalDeaths = Incident::sum('deaths');
        $totalInjured = Incident::sum('injured');
        $totalAffected = Incident::sum('affected');

        // Recent incidents (last 30 days)
        $recentIncidents = Incident::where('created_at', '>=', Carbon::now()->subDays(30))
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Incident trends by month (current year)
        $monthlyTrends = Incident::where('year', $currentYear)
            ->select(
                'month',
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(deaths) as deaths'),
                DB::raw('SUM(injured) as injured')
            )
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->keyBy('month');

        // Fill missing months with zeros
        $monthlyData = [];
        for ($i = 1; $i <= 12; $i++) {
            $monthlyData[] = [
                'month' => Carbon::create()->month($i)->format('M'),
                'count' => $monthlyTrends->get($i)->count ?? 0,
                'deaths' => $monthlyTrends->get($i)->deaths ?? 0,
                'injuries' => $monthlyTrends->get($i)->injured ?? 0,
            ];
        }

        // Incident categories
        $categoriesData = Incident::select('incident_category', DB::raw('COUNT(*) as count'))
            ->whereNotNull('incident_category')
            ->groupBy('incident_category')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // Geographic distribution
        $regionData = Incident::select('region', DB::raw('COUNT(*) as count'))
            ->whereNotNull('region')
            ->groupBy('region')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        $stateData = Incident::select('state', DB::raw('COUNT(*) as count'))
            ->whereNotNull('state')
            ->groupBy('state')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // Severity analysis
        $severityData = Incident::select(
                DB::raw('
                    CASE 
                        WHEN deaths > 10 THEN "Critical"
                        WHEN deaths > 5 THEN "High"
                        WHEN deaths > 0 THEN "Medium"
                        ELSE "Low"
                    END as severity
                '),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('severity')
            ->get();

        // Government involvement statistics
        $governmentStats = [
            'involved' => Incident::where('government_involve', true)->count(),
            'not_involved' => Incident::where('government_involve', false)->count(),
            'military_involved' => Incident::where('military', true)->count(),
        ];

        // Top affected locations
        $topLocations = Incident::select(
                'city_town',
                'state',
                DB::raw('COUNT(*) as incident_count'),
                DB::raw('SUM(deaths) as total_deaths'),
                DB::raw('SUM(injured) as total_injured')
            )
            ->whereNotNull('city_town')
            ->groupBy('city_town', 'state')
            ->orderBy('incident_count', 'desc')
            ->limit(10)
            ->get();

        return view('pages.dashboard', compact(
            'totalIncidents',
            'thisYearIncidents',
            'totalDeaths',
            'totalInjured',
            'totalAffected',
            'recentIncidents',
            'monthlyData',
            'categoriesData',
            'regionData',
            'stateData',
            'severityData',
            'governmentStats',
            'topLocations'
        ));
    }

    public function getIncidentsByDateRange(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->subDays(30));
        $endDate = $request->input('end_date', Carbon::now());

        $incidents = Incident::whereBetween('created_at', [$startDate, $endDate])
            ->select('created_at', 'incident_category', 'deaths', 'injured', 'latitude', 'longitude')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json($incidents);
    }

    public function getGeographicData()
    {
        $data = Incident::select('latitude', 'longitude', 'incident_category', 'deaths', 'injured', 'city_town')
            ->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->get();

        return response()->json($data);
    }
   

    public function clearCache(): View
    {
        Artisan::call('cache:clear');

        return view('clear-cache');
    }
}
