<?php

namespace App\Http\Controllers\API\V4;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\AssetReport;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AssetReportController extends Controller
{
    /**
     * GET Asset Reports based on UserKey and optional LastSyncTime
     * Endpoint: GET /api/v4/asset-reports
     */
    public function getReports(Request $request)
    {
        $key = $request->query('UserKey');
        $lastSyncTime = $request->query('LastSyncTime', null);

        // Validate user
        $user = User::where('id_session', $key)->first();
        if (!$user) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ], 401);
        }

        // Build query
        $query = AssetReport::where('user_id', $user->id)
            ->where('active', 1); // Only get active reports

        // Apply incremental sync if LastSyncTime provided
        if ($lastSyncTime && $lastSyncTime > 0) {
            try {
                $syncDateTime = Carbon::createFromTimestampMs($lastSyncTime);
                $query->where('updated_at', '>', $syncDateTime);
            } catch (\Exception $e) {
                // Invalid timestamp, return all reports
            }
        }

        $reports = $query->orderBy('report_date', 'desc')->get();

        if ($reports->isEmpty()) {
            return response()->json([
                'status' => 'true',
                'errCode' => '02',
                'msg' => 'No Reports Available',
                'data' => []
            ]);
        }

        // Transform data to match Android model
        $transformedReports = $reports->map(function ($report) {
            return [
                'local_id_report' => $report->local_id_report,
                'id_report' => $report->id,
                'state' => $report->state,
                'lga' => $report->lga,
                'ward' => $report->ward,
                'specific_location' => $report->specific_location,
                'latitude' => $report->latitude,
                'longitude' => $report->longitude,
                'asset_type' => $report->asset_type,
                'asset_name' => $report->asset_name,
                'asset_id' => $report->asset_id,
                'situation_status' => $report->situation_status,
                'condition' => $report->condition,
                'completion_percentage' => $report->completion_percentage,
                'damage_level' => $report->damage_level,
                'description' => $report->description,
                'issues' => $report->issues,
                'photo_uris' => $report->photo_uris,
                'user_id' => $report->user_id,
                'report_date' => $report->report_date,
                'active' => $report->active,
                'is_sync' => true,
                'updated_at' => $report->updated_at->timestamp * 1000 // Convert to milliseconds
            ];
        });

        return response()->json($transformedReports);
    }

    /**
     * POST Asset Report Sync - Insert or Update Asset Report data
     * Endpoint: POST /api/v4/asset-reports/sync
     */
    public function syncReport(Request $request)
    {
        $key = $request->input('UserKey');

        // Validate user
        $user = User::where('id_session', $key)->first();
        if (!$user) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ], 401);
        }

        // Validate request data
        $validator = Validator::make($request->all(), [
            'local_id_report' => 'required|string|max:255',
            'state' => 'required|string|max:100',
            'lga' => 'required|string|max:100',
            'ward' => 'required|string|max:100',
            'asset_type' => 'required|string|max:100',
            'asset_name' => 'required|string|max:255',
            'situation_status' => 'required|string|max:100',
            'condition' => 'required|string|max:50',
            'description' => 'required|string',
            'report_date' => 'required|numeric',
            'active' => 'required|integer|in:0,1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'false',
                'errCode' => '04',
                'msg' => 'Validation error',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            DB::beginTransaction();

            // Check if report exists by local_id_report
            $report = AssetReport::where('local_id_report', $request->input('local_id_report'))
                ->first();

            // Prepare data
            $data = [
                'state' => $request->input('state'),
                'lga' => $request->input('lga'),
                'ward' => $request->input('ward'),
                'specific_location' => $request->input('specific_location', ''),
                'latitude' => $request->input('latitude', ''),
                'longitude' => $request->input('longitude', ''),
                'asset_type' => $request->input('asset_type'),
                'asset_name' => $request->input('asset_name'),
                'asset_id' => $request->input('asset_id', ''),
                'situation_status' => $request->input('situation_status'),
                'condition' => $request->input('condition'),
                'completion_percentage' => $request->input('completion_percentage', ''),
                'damage_level' => $request->input('damage_level', ''),
                'description' => $request->input('description'),
                'issues' => $request->input('issues', ''),
                'photo_uris' => $request->input('photo_uris', ''),
                'user_id' => $user->id,
                'report_date' => $request->input('report_date'),
                'active' => $request->input('active', 1),
            ];

            if ($report) {
                // Update existing report
                // Only update if the incoming data is newer
                $incomingTimestamp = $request->input('report_date');
                $existingTimestamp = $report->report_date;

                if ($incomingTimestamp >= $existingTimestamp) {
                    $report->update($data);
                    $message = 'Report updated successfully';
                } else {
                    $message = 'Report already up to date';
                }
            } else {
                // Create new report
                $data['local_id_report'] = $request->input('local_id_report');
                $report = AssetReport::create($data);
                $message = 'Report created successfully';
            }

            DB::commit();

            return response()->json([
                [
                    'status' => 'success',
                    'msg' => $message,
                    'id_report' => $report->id,
                    'local_id_report' => $report->local_id_report
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                [
                    'status' => 'error',
                    'msg' => 'Failed to sync report: ' . $e->getMessage()
                ]
            ], 500);
        }
    }

    /**
     * DELETE Asset Report (Soft Delete)
     * Endpoint: DELETE /api/v4/asset-reports/{localId}
     */
    public function deleteReport(Request $request, $localId)
    {
        $key = $request->query('UserKey');

        // Validate user
        $user = User::where('id_session', $key)->first();
        if (!$user) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ], 401);
        }

        $report = AssetReport::where('local_id_report', $localId)
            ->where('user_id', $user->id)
            ->first();

        if (!$report) {
            return response()->json([
                [
                    'status' => 'error',
                    'msg' => 'Report not found'
                ]
            ], 404);
        }

        // Soft delete
        $report->update(['active' => 0]);

        return response()->json([
            [
                'status' => 'success',
                'msg' => 'Report deleted successfully'
            ]
        ]);
    }

    /**
     * GET Asset Report by ID
     * Endpoint: GET /api/v4/asset-reports/{localId}
     */
    public function getReportById(Request $request, $localId)
    {
        $key = $request->query('UserKey');

        // Validate user
        $user = User::where('id_session', $key)->first();
        if (!$user) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ], 401);
        }

        $report = AssetReport::where('local_id_report', $localId)
            ->where('user_id', $user->id)
            ->where('active', 1)
            ->first();

        if (!$report) {
            return response()->json([
                'status' => 'false',
                'errCode' => '03',
                'msg' => 'Report not found',
            ], 404);
        }

        return response()->json([
            'local_id_report' => $report->local_id_report,
            'id_report' => $report->id,
            'state' => $report->state,
            'lga' => $report->lga,
            'ward' => $report->ward,
            'specific_location' => $report->specific_location,
            'latitude' => $report->latitude,
            'longitude' => $report->longitude,
            'asset_type' => $report->asset_type,
            'asset_name' => $report->asset_name,
            'asset_id' => $report->asset_id,
            'situation_status' => $report->situation_status,
            'condition' => $report->condition,
            'completion_percentage' => $report->completion_percentage,
            'damage_level' => $report->damage_level,
            'description' => $report->description,
            'issues' => $report->issues,
            'photo_uris' => $report->photo_uris,
            'user_id' => $report->user_id,
            'report_date' => $report->report_date,
            'active' => $report->active,
            'is_sync' => true,
            'updated_at' => $report->updated_at->timestamp * 1000
        ]);
    }

    /**
     * GET Reports Statistics
     * Endpoint: GET /api/v4/asset-reports/statistics
     */
    public function getStatistics(Request $request)
    {
        $key = $request->query('UserKey');

        // Validate user
        $user = User::where('id_session', $key)->first();
        if (!$user) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ], 401);
        }

        $totalReports = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->count();

        $reportsThisMonth = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->whereMonth('created_at', Carbon::now()->month)
            ->whereYear('created_at', Carbon::now()->year)
            ->count();

        $reportsThisWeek = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->whereBetween('created_at', [
                Carbon::now()->startOfWeek(),
                Carbon::now()->endOfWeek()
            ])
            ->count();

        $totalAssetTypes = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->distinct('asset_type')
            ->count('asset_type');

        $totalLGAs = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->distinct('lga')
            ->count('lga');

        $totalWards = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->distinct('ward')
            ->count('ward');

        $problematicAssets = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->where(function($query) {
                $query->whereIn('condition', ['Poor', 'Very Poor', 'Dilapidated'])
                    ->orWhereIn('situation_status', ['Damaged', 'Abandoned', 'Needs Maintenance', 'Needs Renovation'])
                    ->orWhere('damage_level', '!=', 'No Damage');
            })
            ->count();

        $operationalAssets = AssetReport::where('user_id', $user->id)
            ->where('active', 1)
            ->where('situation_status', 'Operational')
            ->count();

        return response()->json([
            'total_reports' => $totalReports,
            'reports_this_month' => $reportsThisMonth,
            'reports_this_week' => $reportsThisWeek,
            'total_asset_types' => $totalAssetTypes,
            'total_lgas_covered' => $totalLGAs,
            'total_wards_covered' => $totalWards,
            'problematic_assets_count' => $problematicAssets,
            'operational_assets_count' => $operationalAssets
        ]);
    }

    /**
     * GET Reports by filters
     * Endpoint: GET /api/v4/asset-reports/filter
     */
    public function filterReports(Request $request)
    {
        $key = $request->query('UserKey');
        $lga = $request->query('lga');
        $ward = $request->query('ward');
        $assetType = $request->query('asset_type');
        $situationStatus = $request->query('situation_status');
        $condition = $request->query('condition');

        // Validate user
        $user = User::where('id_session', $key)->first();
        if (!$user) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ], 401);
        }

        $query = AssetReport::where('user_id', $user->id)
            ->where('active', 1);

        if ($lga) {
            $query->where('lga', $lga);
        }

        if ($ward) {
            $query->where('ward', $ward);
        }

        if ($assetType) {
            $query->where('asset_type', $assetType);
        }

        if ($situationStatus) {
            $query->where('situation_status', $situationStatus);
        }

        if ($condition) {
            $query->where('condition', $condition);
        }

        $reports = $query->orderBy('report_date', 'desc')->get();

        $transformedReports = $reports->map(function ($report) {
            return [
                'local_id_report' => $report->local_id_report,
                'id_report' => $report->id,
                'state' => $report->state,
                'lga' => $report->lga,
                'ward' => $report->ward,
                'specific_location' => $report->specific_location,
                'latitude' => $report->latitude,
                'longitude' => $report->longitude,
                'asset_type' => $report->asset_type,
                'asset_name' => $report->asset_name,
                'asset_id' => $report->asset_id,
                'situation_status' => $report->situation_status,
                'condition' => $report->condition,
                'completion_percentage' => $report->completion_percentage,
                'damage_level' => $report->damage_level,
                'description' => $report->description,
                'issues' => $report->issues,
                'photo_uris' => $report->photo_uris,
                'user_id' => $report->user_id,
                'report_date' => $report->report_date,
                'active' => $report->active,
                'is_sync' => true,
                'updated_at' => $report->updated_at->timestamp * 1000
            ];
        });

        return response()->json($transformedReports);
    }
}