<?php

namespace App\Http\Controllers\API\V4;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Incident;
use Illuminate\Support\Facades\Validator;

class IncidentController extends Controller
{
    /**
     * GET Incidents based on UserKey and optional LastSyncTime
     */
    public function getIncidents(Request $request)
    {
        $key = $request->query('UserKey');
        $lastSyncTime = $request->query('LastSyncTime', null);

        $user = User::where('id_session', $key)->first();
        if (!$user) {
            return response()->json([
                'status' => 'true',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ]);
        }

        $query = Incident::where('user_id', $user->id);

        if ($lastSyncTime && $lastSyncTime > 1) {
            $query->where('updated_at', '>', now()->createFromTimestampMs($lastSyncTime));
        }

        $incidents = $query->get();

        if ($incidents->isEmpty()) {
            return response()->json([
                'status' => 'true',
                'errCode' => '02',
                'msg' => 'No Incident Available',
            ]);
        }

        return response()->json($incidents);
    }

    /**
     * POST Incident Sync - Insert or Update Incident data
     */
    public function syncIncident(Request $request)
    {
        $key = $request->input('UserKey');
        $user = User::where('id_session', $key)->first();

        if (!$user) {
            return response()->json([
                'status' => 'false',
                'errCode' => '00',
                'msg' => 'User Not Found',
            ]);
        }

        $validator = Validator::make($request->all(), [
            'local_id' => 'required|string',
            'year' => 'required|integer',
            'incident_category' => 'required|string',
            // Add more validations as needed
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'errCode' => '04',
                'msg' => 'Validation error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $incident = Incident::where('local_id', $request->input('local_id'))->first();

        $data = $request->only([
            'year', 'month', 'day', 'hour', 'region', 'country', 'state', 'local_government',
            'city_town', 'village', 'sub_division_block', 'road_exact_location',
            'latitude', 'longitude', 'incident_category', 'main_victim', 'government_involve',
            'civilian', 'military', 'nsag', 'deaths', 'injured', 'affected', 'perpetrator',
            'source', 'time_before_attack', 'action_taken', 'by_who', 'brief_narrative', 'comment',
        ]);

        $data['user_id'] = $user->id;

        if ($incident) {
            $incident->update($data);
        } else {
            $data['local_id'] = $request->input('local_id');
            Incident::create($data);
        }

        return response()->json([["status" => "success"]]);
    }
}
