<?php

namespace App\Http\Controllers;

use App\Models\AssetReport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class AssetReportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = AssetReport::where('active', 1);

        // Apply filters
        if ($request->filled('lga')) {
            $query->where('lga', $request->lga);
        }

        if ($request->filled('ward')) {
            $query->where('ward', $request->ward);
        }

        if ($request->filled('asset_type')) {
            $query->where('asset_type', $request->asset_type);
        }

        if ($request->filled('situation_status')) {
            $query->where('situation_status', $request->situation_status);
        }

        if ($request->filled('condition')) {
            $query->where('condition', $request->condition);
        }

        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $reports = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('asset_reports.index', compact('reports'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('asset_reports.form');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'lga' => 'required|string|max:255',
            'ward' => 'required|string|max:255',
            'specific_location' => 'nullable|string|max:500',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'asset_type' => 'required|string|max:255',
            'asset_name' => 'required|string|max:255',
            'asset_id' => 'nullable|string|max:100',
            'situation_status' => 'required|string|max:255',
            'condition' => 'required|string|max:255',
            'completion_percentage' => 'nullable|integer|min:0|max:100',
            'damage_level' => 'nullable|string|max:255',
            'description' => 'required|string',
            'issues' => 'nullable|string',
            'photos.*' => 'nullable|image|max:5120', // 5MB max per image
        ]);

        // Generate unique local ID
        $validated['local_id_report'] = (string) Str::uuid();
        $validated['state'] = 'Katsina'; // Fixed state
        $validated['user_id'] = Auth::user()->phone ?? Auth::user()->email;
        $validated['report_date'] = now()->timestamp * 1000; // Milliseconds
        $validated['active'] = 1;
        $validated['is_sync'] = false;

        // Handle photo uploads
        if ($request->hasFile('photos')) {
            $photoUris = [];
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('asset_photos', 'public');
                $photoUris[] = asset('storage/' . $path);
            }
            $validated['photo_uris'] = implode(',', $photoUris);
        }

        AssetReport::create($validated);

        return redirect()->route('asset-reports.index')
            ->with('success', 'Asset report created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(AssetReport $assetReport)
    {
        return view('asset_reports.show', ['report' => $assetReport]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(AssetReport $assetReport)
    {
        return view('asset_reports.form', ['report' => $assetReport]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, AssetReport $assetReport)
    {
        $validated = $request->validate([
            'lga' => 'required|string|max:255',
            'ward' => 'required|string|max:255',
            'specific_location' => 'nullable|string|max:500',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'asset_type' => 'required|string|max:255',
            'asset_name' => 'required|string|max:255',
            'asset_id' => 'nullable|string|max:100',
            'situation_status' => 'required|string|max:255',
            'condition' => 'required|string|max:255',
            'completion_percentage' => 'nullable|integer|min:0|max:100',
            'damage_level' => 'nullable|string|max:255',
            'description' => 'required|string',
            'issues' => 'nullable|string',
            'photos.*' => 'nullable|image|max:5120',
        ]);

        // Mark as unsynced
        $validated['is_sync'] = false;

        // Handle new photo uploads
        if ($request->hasFile('photos')) {
            $photoUris = [];
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('asset_photos', 'public');
                $photoUris[] = asset('storage/' . $path);
            }
            // Append to existing photos
            $existingPhotos = $assetReport->photo_uris ? explode(',', $assetReport->photo_uris) : [];
            $allPhotos = array_merge($existingPhotos, $photoUris);
            $validated['photo_uris'] = implode(',', $allPhotos);
        }

        $assetReport->update($validated);

        return redirect()->route('asset-reports.index')
            ->with('success', 'Asset report updated successfully.');
    }

    /**
     * Remove the specified resource from storage (soft delete).
     */
    public function destroy(AssetReport $assetReport)
    {
        $assetReport->update([
            'active' => 0,
            'is_sync' => false
        ]);

        return redirect()->route('asset-reports.index')
            ->with('success', 'Asset report deleted successfully.');
    }

    /**
     * Export reports to CSV
     */
    public function export(Request $request)
    {
        $query = AssetReport::where('active', 1);

        // Apply same filters as index
        if ($request->filled('lga')) {
            $query->where('lga', $request->lga);
        }
        if ($request->filled('asset_type')) {
            $query->where('asset_type', $request->asset_type);
        }
        if ($request->filled('situation_status')) {
            $query->where('situation_status', $request->situation_status);
        }
        if ($request->filled('condition')) {
            $query->where('condition', $request->condition);
        }

        $reports = $query->orderBy('created_at', 'desc')->get();

        $filename = 'asset_reports_' . date('Y-m-d_His') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function() use ($reports) {
            $file = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($file, [
                'ID', 'Local ID', 'Asset Name', 'Asset Type', 'Asset ID',
                'State', 'LGA', 'Ward', 'Specific Location',
                'Latitude', 'Longitude', 'Situation Status', 'Condition',
                'Completion %', 'Damage Level', 'Description', 'Issues',
                'User ID', 'Report Date', 'Created At', 'Synced'
            ]);

            // Data rows
            foreach ($reports as $report) {
                fputcsv($file, [
                    $report->id,
                    $report->local_id_report,
                    $report->asset_name,
                    $report->asset_type,
                    $report->asset_id,
                    $report->state,
                    $report->lga,
                    $report->ward,
                    $report->specific_location,
                    $report->latitude,
                    $report->longitude,
                    $report->situation_status,
                    $report->condition,
                    $report->completion_percentage,
                    $report->damage_level,
                    $report->description,
                    $report->issues,
                    $report->user_id,
                    date('Y-m-d H:i:s', $report->report_date / 1000),
                    $report->created_at,
                    $report->is_sync ? 'Yes' : 'No'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}