<?php

namespace App\Http\Controllers;

use App\Models\AssetReport;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;
use Illuminate\View\View;

class HomeController extends Controller
{
    public function index(): View
    {
        // Get date range for current year
        $currentYear = Carbon::now()->year;
        $currentMonth = Carbon::now()->month;

        // Basic statistics
        $totalReports = AssetReport::where('active', 1)->count();
        $thisYearReports = AssetReport::where('active', 1)
            ->whereYear('created_at', $currentYear)
            ->count();
        $thisMonthReports = AssetReport::where('active', 1)
            ->whereYear('created_at', $currentYear)
            ->whereMonth('created_at', $currentMonth)
            ->count();

        // Asset condition statistics
        $operationalAssets = AssetReport::where('active', 1)
            ->where('situation_status', 'Operational')
            ->count();
        
        $problematicAssets = AssetReport::where('active', 1)
            ->where(function($query) {
                $query->whereIn('condition', ['Poor', 'Very Poor', 'Dilapidated'])
                    ->orWhereIn('situation_status', ['Damaged', 'Abandoned', 'Needs Maintenance', 'Needs Renovation'])
                    ->orWhere('damage_level', '!=', 'No Damage');
            })
            ->count();

        $abandonedAssets = AssetReport::where('active', 1)
            ->where('situation_status', 'Abandoned')
            ->count();

        $damagedAssets = AssetReport::where('active', 1)
            ->whereIn('situation_status', ['Damaged', 'Needs Maintenance', 'Needs Renovation'])
            ->count();

        // Recent reports (last 30 days)
        $recentReports = AssetReport::where('active', 1)
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Monthly trends (current year)
        $monthlyTrends = AssetReport::where('active', 1)
            ->whereYear('created_at', $currentYear)
            ->select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->keyBy('month');

        // Fill missing months with zeros
        $monthlyData = [];
        for ($i = 1; $i <= 12; $i++) {
            $monthlyData[] = [
                'month' => Carbon::create()->month($i)->format('M'),
                'count' => $monthlyTrends->get($i)->count ?? 0,
            ];
        }

        // Asset types distribution
        $assetTypesData = AssetReport::where('active', 1)
            ->select('asset_type', DB::raw('COUNT(*) as count'))
            ->groupBy('asset_type')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // Geographic distribution (LGAs)
        $lgaData = AssetReport::where('active', 1)
            ->select('lga', DB::raw('COUNT(*) as count'))
            ->groupBy('lga')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // FIXED: Condition distribution with proper ordering
        $conditionData = AssetReport::where('active', 1)
            ->select('condition', DB::raw('COUNT(*) as count'))
            ->groupBy('condition')
            ->get()
            ->sortBy(function($item) {
                // Custom sort order
                $order = [
                    'Excellent' => 1,
                    'Good' => 2,
                    'Fair' => 3,
                    'Poor' => 4,
                    'Very Poor' => 5,
                    'Dilapidated' => 6
                ];
                return $order[$item->condition] ?? 999;
            })
            ->values(); // Reset array keys

        // Situation status distribution
        $statusData = AssetReport::where('active', 1)
            ->select('situation_status', DB::raw('COUNT(*) as count'))
            ->groupBy('situation_status')
            ->orderBy('count', 'desc')
            ->get();

        // Top LGAs with most problematic assets
        $topProblematicLGAs = AssetReport::where('active', 1)
            ->where(function($query) {
                $query->whereIn('condition', ['Poor', 'Very Poor', 'Dilapidated'])
                    ->orWhereIn('situation_status', ['Damaged', 'Abandoned', 'Needs Maintenance']);
            })
            ->select(
                'lga',
                DB::raw('COUNT(*) as problematic_count')
            )
            ->groupBy('lga')
            ->orderBy('problematic_count', 'desc')
            ->limit(10)
            ->get();

        // Coverage statistics
        $totalLGAsCovered = AssetReport::where('active', 1)
            ->distinct('lga')
            ->count('lga');

        $totalWardsCovered = AssetReport::where('active', 1)
            ->distinct('ward')
            ->count('ward');

        $totalAssetTypes = AssetReport::where('active', 1)
            ->distinct('asset_type')
            ->count('asset_type');

        // Active users count
        $activeUsers = User::where('status', 'active')->count();
        $totalUsers = User::count();

        // Damage level statistics
        $damageLevelData = AssetReport::where('active', 1)
            ->where('damage_level', '!=', '')
            ->where('damage_level', '!=', 'No Damage')
            ->select('damage_level', DB::raw('COUNT(*) as count'))
            ->groupBy('damage_level')
            ->orderBy('count', 'desc')
            ->get();

        return view('pages.dashboard', compact(
            'totalReports',
            'thisYearReports',
            'thisMonthReports',
            'operationalAssets',
            'problematicAssets',
            'abandonedAssets',
            'damagedAssets',
            'recentReports',
            'monthlyData',
            'assetTypesData',
            'lgaData',
            'conditionData',
            'statusData',
            'topProblematicLGAs',
            'totalLGAsCovered',
            'totalWardsCovered',
            'totalAssetTypes',
            'activeUsers',
            'totalUsers',
            'damageLevelData'
        ));
    }

    public function getReportsByDateRange(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->subDays(30));
        $endDate = $request->input('end_date', Carbon::now());

        $reports = AssetReport::where('active', 1)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->select('created_at', 'asset_type', 'situation_status', 'condition', 'lga', 'ward', 'asset_name')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json($reports);
    }

    public function getGeographicData()
    {
        $data = AssetReport::where('active', 1)
            ->select('latitude', 'longitude', 'asset_type', 'situation_status', 'condition', 'lga', 'ward', 'asset_name')
            ->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->where('latitude', '!=', '')
            ->where('longitude', '!=', '')
            ->where('latitude', '!=', '0.0')
            ->where('longitude', '!=', '0.0')
            ->get();

        return response()->json($data);
    }

    public function getStatistics()
    {
        $stats = [
            'total_reports' => AssetReport::where('active', 1)->count(),
            'operational' => AssetReport::where('active', 1)->where('situation_status', 'Operational')->count(),
            'problematic' => AssetReport::where('active', 1)
                ->where(function($query) {
                    $query->whereIn('condition', ['Poor', 'Very Poor', 'Dilapidated'])
                        ->orWhereIn('situation_status', ['Damaged', 'Abandoned', 'Needs Maintenance']);
                })
                ->count(),
            'abandoned' => AssetReport::where('active', 1)->where('situation_status', 'Abandoned')->count(),
        ];

        return response()->json($stats);
    }

    public function clearCache(): View
    {
        Artisan::call('cache:clear');
        return view('clear-cache');
    }
}