<!doctype html>
<html class="no-js" lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="x-ua-compatible" content="ie=edge">
    <title>KASAMA - Katsina Asset Situation Monitoring & Analysis</title>
    <meta name="description" content="Katsina State Government Asset Monitoring Dashboard">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    
    <link rel="icon" href="/favicon.png" type="image/x-icon" />
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #2d3748;
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 1000;
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .logo-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 24px;
        }

        .logo-text h1 {
            font-size: 1.5rem;
            font-weight: 700;
            color: #2d3748;
            margin: 0;
        }

        .logo-text p {
            font-size: 0.85rem;
            color: #718096;
            margin: 0;
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            border: none;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.4);
        }

        .btn-secondary {
            background: #e2e8f0;
            color: #4a5568;
        }

        .btn-secondary:hover {
            background: #cbd5e0;
        }

        /* Main Container */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.15);
        }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
        }

        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .stat-icon.blue { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .stat-icon.green { background: linear-gradient(135deg, #48bb78 0%, #38a169 100%); }
        .stat-icon.orange { background: linear-gradient(135deg, #ed8936 0%, #dd6b20 100%); }
        .stat-icon.red { background: linear-gradient(135deg, #f56565 0%, #e53e3e 100%); }
        .stat-icon.purple { background: linear-gradient(135deg, #9f7aea 0%, #805ad5 100%); }
        .stat-icon.teal { background: linear-gradient(135deg, #38b2ac 0%, #319795 100%); }
        .stat-icon.pink { background: linear-gradient(135deg, #ed64a6 0%, #d53f8c 100%); }
        .stat-icon.yellow { background: linear-gradient(135deg, #ecc94b 0%, #d69e2e 100%); }

        .stat-value {
            font-size: 2.5rem;
            font-weight: 700;
            color: #2d3748;
            margin-bottom: 0.5rem;
        }

        .stat-label {
            font-size: 0.9rem;
            color: #718096;
            font-weight: 500;
        }

        .stat-trend {
            font-size: 0.85rem;
            display: flex;
            align-items: center;
            gap: 0.25rem;
            margin-top: 0.5rem;
        }

        .stat-trend.up { color: #48bb78; }
        .stat-trend.down { color: #f56565; }

        /* Content Grid */
        .content-grid {
            display: grid;
            grid-template-columns: repeat(12, 1fr);
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .chart-card {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }

        .chart-card.col-8 { grid-column: span 8; }
        .chart-card.col-4 { grid-column: span 4; }
        .chart-card.col-6 { grid-column: span 6; }
        .chart-card.col-12 { grid-column: span 12; }

        .chart-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }

        .chart-title {
            font-size: 1.25rem;
            font-weight: 600;
            color: #2d3748;
        }

        .chart-subtitle {
            font-size: 0.875rem;
            color: #718096;
            margin-top: 0.25rem;
        }

        /* Recent Reports Table */
        .reports-table {
            width: 100%;
            border-collapse: collapse;
        }

        .reports-table thead {
            background: #f7fafc;
        }

        .reports-table th {
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: #4a5568;
            border-bottom: 2px solid #e2e8f0;
        }

        .reports-table td {
            padding: 1rem;
            border-bottom: 1px solid #e2e8f0;
        }

        .reports-table tbody tr:hover {
            background: #f7fafc;
        }

        .badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .badge.operational { background: #c6f6d5; color: #22543d; }
        .badge.damaged { background: #fed7d7; color: #742a2a; }
        .badge.abandoned { background: #feebc8; color: #7c2d12; }
        .badge.maintenance { background: #fefcbf; color: #744210; }
        .badge.excellent { background: #c6f6d5; color: #22543d; }
        .badge.good { background: #bee3f8; color: #2c5282; }
        .badge.fair { background: #fefcbf; color: #744210; }
        .badge.poor { background: #fed7d7; color: #742a2a; }

        /* Responsive */
        @media (max-width: 1024px) {
            .chart-card.col-8,
            .chart-card.col-4,
            .chart-card.col-6 {
                grid-column: span 12;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }

        @media (max-width: 768px) {
            .container {
                padding: 1rem;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .header-content {
                flex-direction: column;
                gap: 1rem;
            }
            
            .reports-table {
                font-size: 0.875rem;
            }
            
            .reports-table th,
            .reports-table td {
                padding: 0.5rem;
            }
        }

        /* Loading Animation */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(102, 126, 234, 0.3);
            border-radius: 50%;
            border-top-color: #667eea;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>

<body>
    <!-- Header -->
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <div class="logo-icon">
                    <i class="fas fa-building"></i>
                </div>
                <div class="logo-text">
                    <h1>KASAMA</h1>
                    <p>Katsina Asset Situation Monitoring & Analysis</p>
                </div>
            </div>
            
            <div class="header-actions">
                <button class="btn btn-secondary" onclick="location.reload()">
                    <i class="fas fa-sync"></i> Refresh
                </button>
                <a href="/reports" class="btn btn-primary">
                    <i class="fas fa-file-alt"></i> View All Reports
                </a>
                <a href="/logout" class="btn btn-secondary">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </header>

    <!-- Main Container -->
    <div class="container">
        <!-- Stats Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($totalReports) }}</div>
                        <div class="stat-label">Total Reports</div>
                        <div class="stat-trend up">
                            <i class="fas fa-arrow-up"></i>
                            {{ $thisMonthReports }} this month
                        </div>
                    </div>
                    <div class="stat-icon blue">
                        <i class="fas fa-chart-line"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($operationalAssets) }}</div>
                        <div class="stat-label">Operational Assets</div>
                        <div class="stat-trend up">
                            <i class="fas fa-check-circle"></i>
                            {{ $totalReports > 0 ? round(($operationalAssets / $totalReports) * 100) : 0 }}% of total
                        </div>
                    </div>
                    <div class="stat-icon green">
                        <i class="fas fa-check-double"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($problematicAssets) }}</div>
                        <div class="stat-label">Problematic Assets</div>
                        <div class="stat-trend {{ $problematicAssets > 0 ? 'down' : 'up' }}">
                            <i class="fas fa-exclamation-triangle"></i>
                            Needs attention
                        </div>
                    </div>
                    <div class="stat-icon orange">
                        <i class="fas fa-tools"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($abandonedAssets) }}</div>
                        <div class="stat-label">Abandoned Assets</div>
                        <div class="stat-trend down">
                            <i class="fas fa-times-circle"></i>
                            Critical status
                        </div>
                    </div>
                    <div class="stat-icon red">
                        <i class="fas fa-ban"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($totalLGAsCovered) }}</div>
                        <div class="stat-label">LGAs Covered</div>
                        <div class="stat-trend up">
                            <i class="fas fa-map-marker-alt"></i>
                            Out of 34 LGAs
                        </div>
                    </div>
                    <div class="stat-icon purple">
                        <i class="fas fa-map"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($totalWardsCovered) }}</div>
                        <div class="stat-label">Wards Covered</div>
                        <div class="stat-trend up">
                            <i class="fas fa-location-dot"></i>
                            Geographic coverage
                        </div>
                    </div>
                    <div class="stat-icon teal">
                        <i class="fas fa-location-crosshairs"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($totalAssetTypes) }}</div>
                        <div class="stat-label">Asset Types</div>
                        <div class="stat-trend up">
                            <i class="fas fa-layer-group"></i>
                            Categories tracked
                        </div>
                    </div>
                    <div class="stat-icon pink">
                        <i class="fas fa-boxes-stacked"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value">{{ number_format($activeUsers) }}</div>
                        <div class="stat-label">Active Users</div>
                        <div class="stat-trend up">
                            <i class="fas fa-users"></i>
                            {{ $totalUsers }} total users
                        </div>
                    </div>
                    <div class="stat-icon yellow">
                        <i class="fas fa-user-check"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Charts Grid -->
        <div class="content-grid">
            <!-- Monthly Trends -->
            <div class="chart-card col-8">
                <div class="chart-header">
                    <div>
                        <h3 class="chart-title">Monthly Report Trends</h3>
                        <p class="chart-subtitle">Report submissions over the past 12 months</p>
                    </div>
                </div>
                <canvas id="monthlyTrendsChart" height="80"></canvas>
            </div>

            <!-- Condition Distribution -->
            <div class="chart-card col-4">
                <div class="chart-header">
                    <div>
                        <h3 class="chart-title">Asset Condition</h3>
                        <p class="chart-subtitle">Distribution by condition</p>
                    </div>
                </div>
                <canvas id="conditionChart" height="200"></canvas>
            </div>

            <!-- Asset Types -->
            <div class="chart-card col-6">
                <div class="chart-header">
                    <div>
                        <h3 class="chart-title">Asset Types</h3>
                        <p class="chart-subtitle">Top 10 most reported asset types</p>
                    </div>
                </div>
                <canvas id="assetTypesChart" height="100"></canvas>
            </div>

            <!-- LGA Distribution -->
            <div class="chart-card col-6">
                <div class="chart-header">
                    <div>
                        <h3 class="chart-title">LGA Distribution</h3>
                        <p class="chart-subtitle">Top 10 LGAs by number of reports</p>
                    </div>
                </div>
                <canvas id="lgaChart" height="100"></canvas>
            </div>

            <!-- Status Distribution -->
            <div class="chart-card col-6">
                <div class="chart-header">
                    <div>
                        <h3 class="chart-title">Situation Status</h3>
                        <p class="chart-subtitle">Current status of all assets</p>
                    </div>
                </div>
                <canvas id="statusChart" height="100"></canvas>
            </div>

            <!-- Problematic LGAs -->
            <div class="chart-card col-6">
                <div class="chart-header">
                    <div>
                        <h3 class="chart-title">Priority LGAs</h3>
                        <p class="chart-subtitle">LGAs with most problematic assets</p>
                    </div>
                </div>
                <canvas id="problematicLGAChart" height="100"></canvas>
            </div>

            <!-- Recent Reports Table -->
            <div class="chart-card col-12">
                <div class="chart-header">
                    <div>
                        <h3 class="chart-title">Recent Reports</h3>
                        <p class="chart-subtitle">Latest asset reports from the field</p>
                    </div>
                    <a href="/reports" class="btn btn-secondary">
                        View All <i class="fas fa-arrow-right"></i>
                    </a>
                </div>
                
                @if($recentReports->count() > 0)
                <table class="reports-table">
                    <thead>
                        <tr>
                            <th>Asset Name</th>
                            <th>Type</th>
                            <th>Location</th>
                            <th>Status</th>
                            <th>Condition</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($recentReports as $report)
                        <tr>
                            <td><strong>{{ $report->asset_name }}</strong></td>
                            <td>{{ $report->asset_type }}</td>
                            <td>{{ $report->ward }}, {{ $report->lga }}</td>
                            <td>
                                <span class="badge {{ strtolower(str_replace(' ', '-', $report->situation_status)) }}">
                                    {{ $report->situation_status }}
                                </span>
                            </td>
                            <td>
                                <span class="badge {{ strtolower($report->condition) }}">
                                    {{ $report->condition }}
                                </span>
                            </td>
                            <td>{{ \Carbon\Carbon::parse($report->created_at)->format('M d, Y') }}</td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
                @else
                <div style="text-align: center; padding: 3rem; color: #a0aec0;">
                    <i class="fas fa-inbox" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                    <p>No recent reports available</p>
                </div>
                @endif
            </div>
        </div>
    </div>

    <script>
        // Chart.js Configuration
        Chart.defaults.font.family = "'Inter', sans-serif";
        Chart.defaults.color = '#718096';

        // Monthly Trends Chart
        const monthlyCtx = document.getElementById('monthlyTrendsChart').getContext('2d');
        new Chart(monthlyCtx, {
            type: 'line',
            data: {
                labels: {!! json_encode(array_column($monthlyData, 'month')) !!},
                datasets: [{
                    label: 'Reports',
                    data: {!! json_encode(array_column($monthlyData, 'count')) !!},
                    borderColor: '#667eea',
                    backgroundColor: 'rgba(102, 126, 234, 0.1)',
                    fill: true,
                    tension: 0.4,
                    borderWidth: 3
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: '#e2e8f0'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });

        // Condition Distribution Chart
        const conditionCtx = document.getElementById('conditionChart').getContext('2d');
        new Chart(conditionCtx, {
            type: 'doughnut',
            data: {
                labels: {!! json_encode($conditionData->pluck('condition')) !!},
                datasets: [{
                    data: {!! json_encode($conditionData->pluck('count')) !!},
                    backgroundColor: [
                        '#48bb78',
                        '#4299e1',
                        '#ecc94b',
                        '#ed8936',
                        '#f56565',
                        '#c53030'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Asset Types Chart
        const assetTypesCtx = document.getElementById('assetTypesChart').getContext('2d');
        new Chart(assetTypesCtx, {
            type: 'bar',
            data: {
                labels: {!! json_encode($assetTypesData->pluck('asset_type')) !!},
                datasets: [{
                    label: 'Reports',
                    data: {!! json_encode($assetTypesData->pluck('count')) !!},
                    backgroundColor: 'rgba(102, 126, 234, 0.8)'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // LGA Chart
        const lgaCtx = document.getElementById('lgaChart').getContext('2d');
        new Chart(lgaCtx, {
            type: 'bar',
            data: {
                labels: {!! json_encode($lgaData->pluck('lga')) !!},
                datasets: [{
                    label: 'Reports',
                    data: {!! json_encode($lgaData->pluck('count')) !!},
                    backgroundColor: 'rgba(72, 187, 120, 0.8)'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                indexAxis: 'y',
                plugins: {
                    legend: {
                        display: false
                    }
                }
            }
        });

        // Status Chart
        const statusCtx = document.getElementById('statusChart').getContext('2d');
        new Chart(statusCtx, {
            type: 'pie',
            data: {
                labels: {!! json_encode($statusData->pluck('situation_status')) !!},
                datasets: [{
                    data: {!! json_encode($statusData->pluck('count')) !!},
                    backgroundColor: [
                        '#48bb78',
                        '#4299e1',
                        '#ed8936',
                        '#f56565',
                        '#9f7aea',
                        '#38b2ac',
                        '#ed64a6',
                        '#ecc94b'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right'
                    }
                }
            }
        });

        // Problematic LGAs Chart
        const problematicLGACtx = document.getElementById('problematicLGAChart').getContext('2d');
        new Chart(problematicLGACtx, {
            type: 'bar',
            data: {
                labels: {!! json_encode($topProblematicLGAs->pluck('lga')) !!},
                datasets: [{
                    label: 'Problematic Assets',
                    data: {!! json_encode($topProblematicLGAs->pluck('problematic_count')) !!},
                    backgroundColor: 'rgba(245, 101, 101, 0.8)'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                indexAxis: 'y',
                plugins: {
                    legend: {
                        display: false
                    }
                }
            }
        });

        // Auto-refresh every 5 minutes
        setTimeout(() => {
            location.reload();
        }, 300000);
    </script>
</body>
</html>