@extends('layouts.main') 
@section('title', 'KASAMA - Asset Monitoring Dashboard')

@section('content')
    <!-- Enhanced CSS and External Resources -->
    @push('head')
        <link rel="stylesheet" href="{{ asset('plugins/weather-icons/css/weather-icons.min.css') }}">
        <link rel="stylesheet" href="{{ asset('plugins/owl.carousel/dist/assets/owl.carousel.min.css') }}">
        <link rel="stylesheet" href="{{ asset('plugins/owl.carousel/dist/assets/owl.theme.default.min.css') }}">
        <link rel="stylesheet" href="{{ asset('plugins/chartist/dist/chartist.min.css') }}">
        
        <!-- Enhanced Asset Monitoring Dashboard Styles -->
        <style>
            :root {
                --primary-color: #667eea;
                --secondary-color: #764ba2;
                --success-color: #48bb78;
                --warning-color: #ed8936;
                --danger-color: #f56565;
                --info-color: #4299e1;
                --dark-bg: #1a1a1a;
                --card-bg: #ffffff;
                --text-dark: #2d3748;
                --text-light: #718096;
                --shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
                --shadow-hover: 0 8px 25px rgba(0, 0, 0, 0.15);
            }

            /* Enhanced Body Styles */
            body {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            }

            /* Animated Page Header */
            .page-header {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 2rem 0;
                margin-bottom: 2rem;
                border-radius: 10px;
                box-shadow: var(--shadow);
                animation: slideInDown 0.8s ease-out;
            }

            .page-header h1 {
                font-size: 2.5rem;
                font-weight: 700;
                text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
            }

            .page-header .lead {
                font-size: 1.1rem;
                opacity: 0.95;
            }

            /* Enhanced Card Styles */
            .card {
                border: none;
                border-radius: 15px;
                box-shadow: var(--shadow);
                transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
                overflow: hidden;
                margin-bottom: 2rem;
                background: var(--card-bg);
                animation: fadeInUp 0.6s ease-out;
            }

            .card:hover {
                transform: translateY(-5px);
                box-shadow: var(--shadow-hover);
            }

            /* Stat Cards with Gradient Backgrounds */
            .stat-card {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                position: relative;
                overflow: hidden;
                min-height: 140px;
            }

            .stat-card::before {
                content: '';
                position: absolute;
                top: -50%;
                right: -50%;
                width: 100%;
                height: 100%;
                background: rgba(255, 255, 255, 0.1);
                border-radius: 50%;
                transition: all 0.6s ease;
                transform: scale(0);
            }

            .stat-card:hover::before {
                transform: scale(2);
            }

            .stat-card.card-primary {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            }

            .stat-card.card-success {
                background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
            }

            .stat-card.card-warning {
                background: linear-gradient(135deg, #ed8936 0%, #dd6b20 100%);
            }

            .stat-card.card-danger {
                background: linear-gradient(135deg, #f56565 0%, #e53e3e 100%);
            }

            .stat-card.card-info {
                background: linear-gradient(135deg, #4299e1 0%, #3182ce 100%);
            }

            .stat-card.card-purple {
                background: linear-gradient(135deg, #9f7aea 0%, #805ad5 100%);
            }

            .stat-card.card-teal {
                background: linear-gradient(135deg, #38b2ac 0%, #319795 100%);
            }

            .stat-card.card-pink {
                background: linear-gradient(135deg, #ed64a6 0%, #d53f8c 100%);
            }

            /* Animated Counter */
            .counter {
                font-size: 2.8rem;
                font-weight: 700;
                opacity: 0;
                animation: countUp 1.5s ease-out 0.5s forwards;
            }

            .stat-icon {
                font-size: 2.5rem;
                opacity: 0.8;
            }

            .stat-trend {
                font-size: 0.85rem;
                opacity: 0.9;
                margin-top: 0.5rem;
            }

            /* Enhanced Table Styles */
            .table-card {
                border-radius: 15px;
                overflow: hidden;
                background: white;
            }

            .card-header.bg-gradient {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                border: none;
                padding: 1rem 1.5rem;
            }

            .table {
                margin-bottom: 0;
            }

            .table thead th {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                border: none;
                font-weight: 600;
                padding: 1rem;
                font-size: 0.9rem;
            }

            .table tbody tr {
                transition: all 0.3s ease;
            }

            .table tbody tr:hover {
                background-color: rgba(102, 126, 234, 0.1);
                transform: scale(1.01);
            }

            .table tbody td {
                padding: 1rem;
                vertical-align: middle;
            }

            /* Status Badges */
            .badge {
                padding: 0.5rem 1rem;
                border-radius: 20px;
                font-weight: 500;
                text-transform: uppercase;
                font-size: 0.7rem;
                letter-spacing: 0.5px;
            }

            .badge.operational { background: #c6f6d5; color: #22543d; }
            .badge.damaged { background: #fed7d7; color: #742a2a; }
            .badge.abandoned { background: #feebc8; color: #7c2d12; }
            .badge.maintenance { background: #fefcbf; color: #744210; }
            .badge.construction { background: #bee3f8; color: #2c5282; }
            .badge.excellent { background: #c6f6d5; color: #22543d; }
            .badge.good { background: #bee3f8; color: #2c5282; }
            .badge.fair { background: #fefcbf; color: #744210; }
            .badge.poor { background: #fed7d7; color: #742a2a; }

            /* Status Indicators */
            .status-indicator {
                width: 12px;
                height: 12px;
                border-radius: 50%;
                display: inline-block;
                animation: pulse 2s infinite;
            }

            .status-online {
                background: var(--success-color);
                box-shadow: 0 0 0 0 rgba(72, 187, 120, 1);
            }

            .status-warning {
                background: var(--warning-color);
                box-shadow: 0 0 0 0 rgba(237, 137, 54, 1);
            }

            .status-offline {
                background: var(--danger-color);
                box-shadow: 0 0 0 0 rgba(245, 101, 101, 1);
            }

            /* Progress Bars */
            .progress {
                height: 8px;
                border-radius: 4px;
                overflow: hidden;
                background-color: rgba(0, 0, 0, 0.1);
            }

            .progress-bar {
                background: linear-gradient(45deg, #667eea, #764ba2);
                transition: width 1.5s ease-in-out;
                animation: progressFill 2s ease-in-out;
            }

            /* Chart Container */
            .chart-container {
                position: relative;
                height: 300px;
            }

            /* Location Items */
            .location-item {
                transition: all 0.3s ease;
                border: 1px solid rgba(102, 126, 234, 0.2) !important;
            }

            .location-item:hover {
                transform: translateX(5px);
                border-color: rgba(102, 126, 234, 0.5) !important;
                box-shadow: 0 4px 12px rgba(102, 126, 234, 0.2);
            }

            /* Category Items */
            .category-item {
                opacity: 0;
                animation: slideInLeft 0.6s ease-out forwards;
            }

            /* Asset Type Cards */
            .asset-type-card {
                padding: 1rem;
                border-radius: 10px;
                background: rgba(102, 126, 234, 0.05);
                border: 1px solid rgba(102, 126, 234, 0.1);
                transition: all 0.3s ease;
            }

            .asset-type-card:hover {
                background: rgba(102, 126, 234, 0.1);
                transform: translateY(-3px);
            }

            /* Animations */
            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translate3d(0, 40px, 0);
                }
                to {
                    opacity: 1;
                    transform: none;
                }
            }

            @keyframes slideInDown {
                from {
                    opacity: 0;
                    transform: translate3d(0, -40px, 0);
                }
                to {
                    opacity: 1;
                    transform: none;
                }
            }

            @keyframes slideInLeft {
                from {
                    opacity: 0;
                    transform: translateX(-20px);
                }
                to {
                    opacity: 1;
                    transform: translateX(0);
                }
            }

            @keyframes countUp {
                from {
                    opacity: 0;
                    transform: translateY(20px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }

            @keyframes pulse {
                0% {
                    transform: scale(0.95);
                    box-shadow: 0 0 0 0 rgba(72, 187, 120, 0.7);
                }
                70% {
                    transform: scale(1);
                    box-shadow: 0 0 0 10px rgba(72, 187, 120, 0);
                }
                100% {
                    transform: scale(0.95);
                    box-shadow: 0 0 0 0 rgba(72, 187, 120, 0);
                }
            }

            @keyframes progressFill {
                from { width: 0%; }
            }

            /* Responsive Design */
            @media (max-width: 768px) {
                .counter {
                    font-size: 2rem;
                }
                .card {
                    margin-bottom: 1rem;
                }
                .page-header h1 {
                    font-size: 1.8rem;
                }
                .stat-card {
                    min-height: 120px;
                }
            }

            /* Utility Classes */
            .text-gradient {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
            }
        </style>
    @endpush

<div class="container-fluid">
    <!-- Enhanced Header -->
    <div class="page-header text-center">
        <div class="row align-items-center">
            <div class="col-12">
                <h1 class="mb-0">
                    <i class="fas fa-building me-3"></i>
                    {{ __('KASAMA Asset Monitoring Dashboard') }}
                </h1>
                <p class="lead mt-2 mb-0">{{ __('Katsina State Government Asset Situation Monitoring & Analysis') }}</p>
            </div>
        </div>
    </div>

    <!-- Enhanced Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-primary">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $totalReports }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('Total Reports') }}</p>
                            <small class="stat-trend">
                                <i class="fas fa-arrow-up"></i> {{ $thisMonthReports }} this month
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-chart-line stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-success">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $operationalAssets }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('Operational Assets') }}</p>
                            <small class="stat-trend">
                                <span class="status-indicator status-online me-1"></span>
                                {{ $totalReports > 0 ? round(($operationalAssets / $totalReports) * 100) : 0 }}% of total
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-check-double stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-warning">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $problematicAssets }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('Problematic Assets') }}</p>
                            <small class="stat-trend">
                                <i class="fas fa-exclamation-triangle"></i>
                                Needs attention
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-tools stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-danger">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $abandonedAssets }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('Abandoned Assets') }}</p>
                            <small class="stat-trend">
                                <span class="status-indicator status-offline me-1"></span>
                                Critical status
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-ban stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Secondary Statistics Row -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-purple">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $totalLGAsCovered }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('LGAs Covered') }}</p>
                            <small class="stat-trend">
                                <i class="fas fa-map-marker-alt"></i> Out of 34 LGAs
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-map stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-teal">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $totalWardsCovered }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('Wards Covered') }}</p>
                            <small class="stat-trend">
                                <i class="fas fa-location-dot"></i> Geographic coverage
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-location-crosshairs stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-pink">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $totalAssetTypes }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('Asset Types') }}</p>
                            <small class="stat-trend">
                                <i class="fas fa-layer-group"></i> Categories tracked
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-boxes-stacked stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card stat-card card-info">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-8">
                            <h4 class="counter mb-0" data-count="{{ $activeUsers }}">0</h4>
                            <p class="mb-0 text-white-50">{{ __('Active Reporters') }}</p>
                            <small class="stat-trend">
                                <i class="fas fa-users"></i> {{ $totalUsers }} total users
                            </small>
                        </div>
                        <div class="col-4 text-end">
                            <i class="fas fa-user-check stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Enhanced Analytics Row -->
    <div class="row mb-4">
        <!-- Monthly Trends Chart -->
        <div class="col-lg-8 mb-4">
            <div class="card h-100">
                <div class="card-header bg-gradient">
                    <h5 class="mb-0">
                        <i class="fas fa-chart-area me-2"></i>
                        {{ __('Monthly Report Trends') }}
                    </h5>
                    <small class="text-white-50">Report submissions over the past 12 months</small>
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="monthlyTrendChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Asset Condition Distribution -->
        <div class="col-lg-4 mb-4">
            <div class="card h-100">
                <div class="card-header bg-gradient">
                    <h5 class="mb-0">
                        <i class="fas fa-chart-pie me-2"></i>
                        {{ __('Asset Condition') }}
                    </h5>
                    <small class="text-white-50">Distribution by condition</small>
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="conditionChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Asset Types and LGAs Row -->
    <div class="row mb-4">
        <!-- Top Asset Types -->
        <div class="col-lg-6 mb-4">
            <div class="card h-100">
                <div class="card-header bg-gradient">
                    <h5 class="mb-0">
                        <i class="fas fa-layer-group me-2"></i>
                        {{ __('Top Asset Types') }}
                    </h5>
                    <small class="text-white-50">Most reported asset categories</small>
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="assetTypesChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Top LGAs -->
        <div class="col-lg-6 mb-4">
            <div class="card h-100">
                <div class="card-header bg-gradient">
                    <h5 class="mb-0">
                        <i class="fas fa-map me-2"></i>
                        {{ __('LGA Distribution') }}
                    </h5>
                    <small class="text-white-50">Top 10 LGAs by number of reports</small>
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="lgaChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Status and Priority Row -->
    <div class="row mb-4">
        <!-- Situation Status -->
        <div class="col-lg-6 mb-4">
            <div class="card h-100">
                <div class="card-header bg-gradient">
                    <h5 class="mb-0">
                        <i class="fas fa-info-circle me-2"></i>
                        {{ __('Situation Status') }}
                    </h5>
                    <small class="text-white-50">Current status of all assets</small>
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="statusChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Priority LGAs -->
        <div class="col-lg-6 mb-4">
            <div class="card h-100">
                <div class="card-header bg-gradient">
                    <h5 class="mb-0">
                        <i class="fas fa-exclamation-circle me-2"></i>
                        {{ __('Priority LGAs') }}
                    </h5>
                    <small class="text-white-50">LGAs with most problematic assets</small>
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="problematicLGAChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Recent Reports and Top Locations Row -->
    <div class="row">
        <!-- Recent Reports Table -->
        <div class="col-lg-8 mb-4">
            <div class="card table-card">
                <div class="card-header bg-gradient d-flex justify-content-between align-items-center">
                    <div>
                        <h5 class="mb-0">
                            <i class="fas fa-list me-2"></i>
                            {{ __('Recent Asset Reports') }}
                        </h5>
                        <small class="text-white-50">Latest reports from the field</small>
                    </div>
                    <a href="{{ route('reports.index') }}" class="btn btn-light btn-sm">
                        <i class="fas fa-external-link-alt me-1"></i>
                        {{ __('View All') }}
                    </a>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>{{ __('Asset Name') }}</th>
                                    <th>{{ __('Type') }}</th>
                                    <th>{{ __('Location') }}</th>
                                    <th>{{ __('Status') }}</th>
                                    <th>{{ __('Condition') }}</th>
                                    <th>{{ __('Date') }}</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($recentReports as $report)
                                <tr>
                                    <td>
                                        <div class="fw-medium">{{ $report->asset_name }}</div>
                                        @if($report->asset_id)
                                        <small class="text-muted">ID: {{ $report->asset_id }}</small>
                                        @endif
                                    </td>
                                    <td>
                                        <span class="badge bg-info">
                                            {{ $report->asset_type }}
                                        </span>
                                    </td>
                                    <td>
                                        <div>
                                            <div class="fw-medium">{{ $report->ward }}</div>
                                            <small class="text-muted">{{ $report->lga }}</small>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge {{ strtolower(str_replace(' ', '-', $report->situation_status)) }}">
                                            {{ $report->situation_status }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge {{ strtolower(str_replace(' ', '-', $report->condition)) }}">
                                            {{ $report->condition }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="text-muted">
                                            {{ \Carbon\Carbon::parse($report->created_at)->format('M d, Y') }}
                                        </div>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="6" class="text-center py-4">
                                        <i class="fas fa-inbox text-muted me-2"></i>
                                        {{ __('No recent reports available') }}
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Most Affected LGAs -->
        <div class="col-lg-4 mb-4">
            <div class="card h-100">
                <div class="card-header bg-gradient">
                    <h5 class="mb-0">
                        <i class="fas fa-map-marked-alt me-2"></i>
                        {{ __('Priority Locations') }}
                    </h5>
                    <small class="text-white-50">LGAs requiring immediate attention</small>
                </div>
                <div class="card-body">
                    @forelse($topProblematicLGAs as $location)
                    <div class="location-item mb-3 p-3 border rounded">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <div>
                                <h6 class="mb-0 fw-bold">{{ $location->lga }}</h6>
                                <small class="text-muted">Katsina State</small>
                            </div>
                            <span class="badge bg-danger fs-6">{{ $location->problematic_count }}</span>
                        </div>
                        <div class="mb-2">
                            <small class="text-muted">
                                <i class="fas fa-exclamation-triangle text-danger me-1"></i>
                                Problematic Assets
                            </small>
                        </div>
                        <div class="progress" style="height: 6px;">
                            <div class="progress-bar bg-danger" 
                                 style="width: {{ ($location->problematic_count / $topProblematicLGAs->max('problematic_count')) * 100 }}%"></div>
                        </div>
                    </div>
                    @empty
                    <div class="text-center py-4">
                        <i class="fas fa-check-circle text-success fa-3x mb-3"></i>
                        <p class="text-muted">{{ __('No problematic assets reported') }}</p>
                    </div>
                    @endforelse
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

<!-- Enhanced JavaScript -->
@push('script')
    <script src="{{ asset('plugins/owl.carousel/dist/owl.carousel.min.js') }}"></script>
    <script src="{{ asset('plugins/chartist/dist/chartist.min.js') }}"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <script>
        // Animated Counter Function
        function animateCounter() {
            const counters = document.querySelectorAll('.counter');
            counters.forEach(counter => {
                const target = parseInt(counter.getAttribute('data-count'));
                const duration = 2000;
                const step = target / (duration / 16);
                let current = 0;
                
                const updateCounter = () => {
                    current += step;
                    if (current < target) {
                        counter.textContent = Math.floor(current).toLocaleString();
                        requestAnimationFrame(updateCounter);
                    } else {
                        counter.textContent = target.toLocaleString();
                    }
                };
                updateCounter();
            });
        }

        // Initialize counters on page load
        window.addEventListener('load', animateCounter);

        // Chart.js Configuration
        Chart.defaults.font.family = "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif";
        Chart.defaults.color = '#718096';

        // Monthly Trends Chart
        const monthlyCtx = document.getElementById('monthlyTrendChart');
        if (monthlyCtx) {
            new Chart(monthlyCtx, {
                type: 'line',
                data: {
                    labels: {!! json_encode(array_column($monthlyData, 'month')) !!},
                    datasets: [{
                        label: 'Reports',
                        data: {!! json_encode(array_column($monthlyData, 'count')) !!},
                        borderColor: '#667eea',
                        backgroundColor: 'rgba(102, 126, 234, 0.1)',
                        fill: true,
                        tension: 0.4,
                        borderWidth: 3,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            padding: 12,
                            titleFont: { size: 14 },
                            bodyFont: { size: 13 }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: { color: '#e2e8f0' },
                            ticks: { font: { size: 11 } }
                        },
                        x: {
                            grid: { display: false },
                            ticks: { font: { size: 11 } }
                        }
                    }
                }
            });
        }

        // Condition Distribution Chart
        const conditionCtx = document.getElementById('conditionChart');
        if (conditionCtx) {
            new Chart(conditionCtx, {
                type: 'doughnut',
                data: {
                    labels: {!! json_encode($conditionData->pluck('condition')) !!},
                    datasets: [{
                        data: {!! json_encode($conditionData->pluck('count')) !!},
                        backgroundColor: [
                            '#48bb78', '#4299e1', '#ecc94b', 
                            '#ed8936', '#f56565', '#c53030'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: { padding: 15, font: { size: 11 } }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            padding: 12
                        }
                    }
                }
            });
        }

        // Asset Types Chart
        const assetTypesCtx = document.getElementById('assetTypesChart');
        if (assetTypesCtx) {
            new Chart(assetTypesCtx, {
                type: 'bar',
                data: {
                    labels: {!! json_encode($assetTypesData->pluck('asset_type')) !!},
                    datasets: [{
                        label: 'Reports',
                        data: {!! json_encode($assetTypesData->pluck('count')) !!},
                        backgroundColor: 'rgba(102, 126, 234, 0.8)',
                        borderRadius: 6
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { display: false } },
                    scales: {
                        y: { 
                            beginAtZero: true,
                            grid: { color: '#e2e8f0' }
                        },
                        x: {
                            grid: { display: false },
                            ticks: { font: { size: 10 } }
                        }
                    }
                }
            });
        }

        // LGA Chart
        const lgaCtx = document.getElementById('lgaChart');
        if (lgaCtx) {
            new Chart(lgaCtx, {
                type: 'bar',
                data: {
                    labels: {!! json_encode($lgaData->pluck('lga')) !!},
                    datasets: [{
                        label: 'Reports',
                        data: {!! json_encode($lgaData->pluck('count')) !!},
                        backgroundColor: 'rgba(72, 187, 120, 0.8)',
                        borderRadius: 6
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    indexAxis: 'y',
                    plugins: { legend: { display: false } },
                    scales: {
                        x: { 
                            beginAtZero: true,
                            grid: { color: '#e2e8f0' }
                        },
                        y: {
                            grid: { display: false },
                            ticks: { font: { size: 10 } }
                        }
                    }
                }
            });
        }

        // Status Chart
        const statusCtx = document.getElementById('statusChart');
        if (statusCtx) {
            new Chart(statusCtx, {
                type: 'pie',
                data: {
                    labels: {!! json_encode($statusData->pluck('situation_status')) !!},
                    datasets: [{
                        data: {!! json_encode($statusData->pluck('count')) !!},
                        backgroundColor: [
                            '#48bb78', '#4299e1', '#ed8936', '#f56565',
                            '#9f7aea', '#38b2ac', '#ed64a6', '#ecc94b'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right',
                            labels: { padding: 15, font: { size: 11 } }
                        }
                    }
                }
            });
        }

        // Problematic LGAs Chart
        const problematicLGACtx = document.getElementById('problematicLGAChart');
        if (problematicLGACtx) {
            new Chart(problematicLGACtx, {
                type: 'bar',
                data: {
                    labels: {!! json_encode($topProblematicLGAs->pluck('lga')) !!},
                    datasets: [{
                        label: 'Problematic Assets',
                        data: {!! json_encode($topProblematicLGAs->pluck('problematic_count')) !!},
                        backgroundColor: 'rgba(245, 101, 101, 0.8)',
                        borderRadius: 6
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    indexAxis: 'y',
                    plugins: { legend: { display: false } },
                    scales: {
                        x: { 
                            beginAtZero: true,
                            grid: { color: '#e2e8f0' }
                        },
                        y: {
                            grid: { display: false },
                            ticks: { font: { size: 10 } }
                        }
                    }
                }
            });
        }

        // Auto-refresh every 5 minutes
        setTimeout(() => {
            location.reload();
        }, 300000);

        // Add smooth scroll behavior
        document.documentElement.style.scrollBehavior = 'smooth';
    </script>
@endpush